/*
 *  metadata.js
 *
 *  Copyright: (c) 2018 FileMaker, Inc. All rights reserved.
 *
 *  FILEMAKER CONFIDENTIAL
 *  This file includes unpublished proprietary source code of FileMaker,
 *  Inc. The copyright notice above does not evidence any actual or
 *  intended publication of such source code.
 */
'use strict';
var errorHandler = require('../routes/error');
var util = require('../routes/util');
var basicAuth = require('basic-auth');

/**
 * @api {GET} /databases Get Database List
 * @apiDescription Get all databases visible to user.
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of databases
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "records": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases
 */
module.exports.dbNames = function(req, res, next) {
	 var ipStr = util.parseClientIp(req);
	 var params = {
		 'requestMethod': req.method,
		 'requestPath': req.originalUrl,
		 'requestIp': ipStr,
		 'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		 'version': req.swagger.params.version.value
		};

	// Database list requires login credentials
	if (req.swagger.params.hasOwnProperty('Authorization') && req.swagger.params.Authorization.value) {
		var authorization = req.swagger.params.Authorization.value;
		var auth = basicAuth(req);
		if (!auth) {
			if (util.cloudVersion() === 2) {
				if (authorization.toLowerCase().startsWith("fmid ")) {
					params.fmidToken = authorization.split(" ")[1];
				} else {
					return errorHandler.handleError('Unauthorized', req, res, "HTTP Authorization header must include FMID token.","10");
				}
			} else {
				if ((req.swagger.params.hasOwnProperty('X-FM-Data-OAuth-Request-Id') && req.swagger.params['X-FM-Data-OAuth-Request-Id'].value) &&
				(req.swagger.params.hasOwnProperty('X-FM-Data-OAuth-Identifier') && req.swagger.params['X-FM-Data-OAuth-Identifier'].value)) {
					params.oAuthRequestId = req.swagger.params['X-FM-Data-OAuth-Request-Id'].value;
					params.oAuthIdentifier = req.swagger.params['X-FM-Data-OAuth-Identifier'].value;
				} else {
					return errorHandler.handleError('Unauthorized', req, res, "HTTP Authorization header invalid","960");
				}
			}
		} else {
			if (util.cloudVersion() === 2) {
				return errorHandler.handleError('Unauthorized', req, res, "HTTP Authorization header must include FMID token.","10");
			} else {
				params.username = auth.name;
				params.password = auth.pass;
			}
		}
	}
	else
	{
		if (util.cloudVersion() === 2) {
			return errorHandler.handleError('Unauthorized', req, res, "HTTP Authorization header is missing.","10");
		}
	}
	try{
		 util.thrift_client.databaseList(params,
			 function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
		});
	 } catch (err){
		 util.thriftExceptionHandler(err, req, res);
	 }
};

/**
 * @api {GET} /databases/:database/layouts Get Layout List
 * @apiDescription Get list of layouts for :database.
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of layouts for :database
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "records": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases
 */
module.exports.layoutNames = function(req, res, next) {
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	   };

	var result = util.setAuthentication(req, res, params, false);	
	if(result){
		try{
			util.thrift_client.layoutList(params,
				function(thrifError, thrifResult){
					return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
				util.thriftExceptionHandler(err, req, res);
		}
	}
};

/**
 * @api {GET} /databases/:database/scripts Get Script List
 * @apiDescription Get list of scripts for :database.
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of layouts for :database
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "records": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases
 */
module.exports.scriptNames = function(req, res, next) {
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	   };

   var result = util.setAuthentication(req, res, params, false);	
   if(result){
		try{
			util.thrift_client.scriptNames(params,
			function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};

/**
 * @api {GET} /databases/:database/layouts/:layout Get Script List
 * @apiDescription Get list of scripts for :database.
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of layouts for :database
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "records": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases
 */
module.exports.layoutMetadata = function(req, res, next) {
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	   };

	var result = util.setAuthentication(req, res, params, false);	
	if(result){   
		if (req.swagger.params.recordId && req.swagger.params.recordId.value)
		{
			params.recordId = req.swagger.params.recordId.value;
		}

		try{
			util.thrift_client.fullLayoutMetaData(params,
				function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
		});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};

/**
 * @api {GET} /databases/:database/layouts/:layout/metadata Get Script List
 * @apiDescription Get old layout metadata format.
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of layouts for :database
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "records": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases
 */
module.exports.oldLayoutMetadata = function(req, res, next) {
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
	   };

   var result = util.setAuthentication(req, res, params, false);	
   if(result){
		try{
			util.thrift_client.layoutMetaData(params,
			function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};
